import { FormItem, FormContainer } from '@/components/ui/Form'
import Input from '@/components/ui/Input'
import Button from '@/components/ui/Button'
import Select from '@/components/ui/Select'
import Radio from '@/components/ui/Radio'
import Upload from '@/components/ui/Upload'
import * as Yup from 'yup'
import { Field, Form, Formik, useFormikContext } from 'formik'
import type { FieldInputProps, FieldProps } from 'formik'
import { HiCheckCircle } from 'react-icons/hi'
import { useAppSelector } from '@/store'
import { useNavigate } from 'react-router-dom'
import useCustomToastify from '@/utils/customHook/useCustomToastify'
import { ComponentType, useEffect, useState } from 'react'
import { InputGroup } from '@/components/ui'
import { components, OptionProps, SingleValueProps } from 'react-select'
import { countryList } from '@/constants/countries.constant'
import { NumericFormat, NumericFormatProps } from 'react-number-format'
import type { InputProps } from '@/components/ui/Input'
import { FcImageFile } from 'react-icons/fc'
import { FetchData } from '@/utils/API Fech/Fetch Function/state_countyFetchFun'

const { SingleValue } = components

type CountryOption = {
    label: string
    dialCode: string
    value: string
}

type stateOptionAll = {
    value: number,
    label: string
}

const userRoles = import.meta.env.VITE_USER_ROLL.split(",").map((role: string, index: any) => {
    return {
        value: index,
        label: role.split("_").map((word: string) => word.charAt(0).toUpperCase() + word.slice(1)).join(" "),
    }
})

const statusOptions = [
    { value: 'active', label: 'Active' },
    { value: 'inactive', label: 'Inactive' },
]

const PhoneSelectOption = ({
    innerProps,
    data,
    isSelected,
}: OptionProps<CountryOption>) => {
    return (
        <div
            className={`cursor-pointer flex items-center justify-between p-2  w-60 ${isSelected
                ? 'bg-gray-100 dark:bg-gray-500'
                : 'hover:bg-gray-50 dark:hover:bg-gray-600'
                }`}
            {...innerProps}
        >
            <div className="flex items-center gap-2">
                <span>
                    ({data.label}) {data.dialCode}
                </span>
            </div>
        </div>
    )
}

const PhoneControl = (props: SingleValueProps<CountryOption>) => {
    const selected = props.getValue()[0]
    return (
        <SingleValue {...props}>
            {selected && <span>{selected.value} {selected.dialCode}</span>}
        </SingleValue>
    )
}

const NumericFormatInput = ({
    onValueChange,
    ...rest
}: Omit<NumericFormatProps, 'form'> & {
    /* eslint-disable @typescript-eslint/no-explicit-any */
    form: any
    field: FieldInputProps<unknown>
}) => {
    return (
        <NumericFormat
            customInput={Input as ComponentType}
            type="text"
            autoComplete="off"
            onValueChange={onValueChange}
            {...rest}
        />
    )
}

const NumberInput = (props: InputProps) => {
    return <Input {...props} value={props.field.value} />
}

const validationSchema = Yup.object().shape({
    picture: Yup.array().min(1, 'At least one file uploaded!'),
    status: Yup.string().required('Status is required'),
    firstName: Yup.string().required('First name is required'),
    lastName: Yup.string().required('Last name is required'),
    address: Yup.object().shape({
        street1: Yup.string().required('Street 1 is required'),
        street2: Yup.string(),
        city: Yup.string().required('City is required'),
        state: Yup.string().required('State is required'),
        country: Yup.string().required('Country is required'),
        zipCode: Yup.string().required('Zip code is required'),
    }),
    email: Yup.string().email('Invalid email address').required('Email address is required'),
    // mobile: Yup.string().required('Mobile number is required'),
    userRole: Yup.string().required('User role is required'),
    // sex: Yup.string().required('Sex is required'),
    dialCode: Yup.string().required('Please select dial code'),
    phoneNumber: Yup.string().required('Please enter your phone number'),
})

// Define initial values
const initialValues = {
    picture: [],
    status: '',
    firstName: '',
    lastName: '',
    address: {
        street1: '',
        street2: '',
        city: '',
        state: '',
        country: '',
        zipCode: '',
    },
    email: '',
    dialCode: '+91',
    phoneNumber: '',
    userRole: '',
    // sex: '',
}

interface valuesParameter {
    firstName: string,
    lastName: string,
    picture: any,
    status: string,
    email: string,
    dialCode: string,
    phoneNumber: string,
    userRole: string,
    address: {
        street1: string,
        street2: string,
        city: string,
        state: string,
        country: string,
        zipCode: string
    }
}


const UserForm = () => {
    const theme = useAppSelector((state) => state.theme)
    const navigate = useNavigate()
    const customToast = useCustomToastify()

    const [countryName, setCountryName] = useState<stateOptionAll[]>([])
    const [stateName, setStateName] = useState<stateOptionAll[]>([])
    const [cityName, setCityName] = useState<stateOptionAll[]>([])



    useEffect(() => {
        getCountryList()
    }, [])



    const getCountryList: any = async () => {
        const data = await FetchData("country_list")
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setCountryName((prev) => [...tempData])
    }

    const getStateList: any = async (countryId: number) => {
        const data = await FetchData(`state_list?country_id=${countryId}`)
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setStateName((prev) => [...tempData])
    }
    const getCityList: any = async (stateId: number) => {
        const data = await FetchData(`city_list?state_id=${stateId}`)
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setCityName((prev) => [...tempData])
    }

    const storeUserDeatils: any = async (values: valuesParameter, setSubmitting: any, resetForm: any) => {
        try {
            const formData = new FormData()
            formData.append('first_name', values.firstName)
            formData.append('last_name', values.lastName)
            formData.append('picture', values.picture[0])
            formData.append('country', values.address.country)
            formData.append('state', values.address.state)
            formData.append('city', values.address.city)
            formData.append('street_1', values.address.street1)
            formData.append('street_2', values.address.street2)
            formData.append('zip_code', values.address.zipCode)
            formData.append('email', values.email)
            formData.append('mobile', values.phoneNumber)
            formData.append('user_role', values.userRole)
            formData.append('status', (values.status === 'active' ? 1 : 0).toString())
            formData.append('dial_code', values.dialCode)

            // for (const pair of formData.entries()) {
            //     console.log(pair[0], pair[1]);
            // }
            // return

            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/users/add-user`, {
                method: 'POST',
                body: formData
            })
            const response = await api.json()
            if (!response.error) {
                customToast.success("User Deatils Added Successfully!")
                setSubmitting(false)
                resetForm()
                navigate('/auth/users/allUsers')
            } else {
                customToast.danger(response.message)
            }
        } catch (error) {
            console.error(error)
        }
    }


    const beforeUpload = (file: FileList | null, fileList: File[]) => {
        let valid: string | boolean = true

        const allowedFileType = ['image/jpeg', 'image/png']
        const MAX_FILE_SIZE = 500000

        if (fileList.length > 1) {
            return `You can only upload one file`
        }

        if (file) {
            for (const f of file) {
                if (!allowedFileType.includes(f.type)) {
                    valid = 'Please upload a .jpeg or .png file!'
                }

                if (f.size >= MAX_FILE_SIZE) {
                    valid = 'Upload image cannot be more than 500kb!'
                }
            }
        }

        return valid
    }

    return (
        <>
            <div className={` p-3 rounded-2xl`} >
                <h2 className='mt-3 mb-5 ml-3'>Add User Details</h2>

                <div className='px-3'>
                    <Formik
                        initialValues={initialValues}
                        validationSchema={validationSchema}
                        onSubmit={(values, { resetForm, setSubmitting }) => {
                            try {
                                // console.log("jen", values)
                                // setSubmitting(false)
                                // resetForm()
                                storeUserDeatils(values, setSubmitting, resetForm)

                            } catch (error) {
                                console.log("Error: ", error)
                            }
                        }}
                    >
                        {({ values, touched, errors, resetForm }) => (
                            <Form>
                                {/* style={{ gridTemplateColumns: '60% 28%' }} */}
                                <FormContainer>
                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-7 "  >
                                        <div className={`border rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                            <div className={`text-2xl mb-5 h-fit rounded-t-md bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Basic Deatils</div>
                                            <div className='p-2 px-4'>
                                                <FormItem asterisk
                                                    label="First Name"
                                                    invalid={errors.firstName && touched.firstName}
                                                    errorMessage={errors.firstName}>
                                                    <Field
                                                        type="text"
                                                        name="firstName"
                                                        placeholder="First Name"
                                                        component={Input}
                                                        size='md' />
                                                </FormItem>

                                                <FormItem asterisk label="Last Name" invalid={errors.lastName && touched.lastName} errorMessage={errors.lastName}>
                                                    <Field type="text" name="lastName" placeholder="Last Name" component={Input} size='md' />
                                                </FormItem>
                                                <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-4">
                                                    <FormItem asterisk label="User Role" invalid={errors.userRole && touched.userRole} errorMessage={errors.userRole}>
                                                        <Field name="userRole" >
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Select
                                                                    field={field}
                                                                    form={form}
                                                                    size='md'
                                                                    options={userRoles}
                                                                    value={userRoles.find((option: any) => option.value === values.userRole)}
                                                                    onChange={(option) => form.setFieldValue(field.name, option?.value)}
                                                                />
                                                            )}
                                                        </Field>
                                                    </FormItem>

                                                    {/* <FormItem asterisk label="Sex" invalid={errors.sex && touched.sex} errorMessage={errors.sex}>
                                                        <Field name="sex">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Radio.Group value={values.sex} onChange={(val) => form.setFieldValue(field.name, val)}>
                                                                    <Radio value="male" >Male</Radio>
                                                                    <Radio value="female">Female</Radio>
                                                                </Radio.Group>
                                                            )}
                                                        </Field>
                                                    </FormItem> */}
                                                </div>
                                            </div>
                                        </div>
                                        <div>
                                            <div className={`border rounded-md h-full ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit rounded-t-md bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Profile Image</div>
                                                <div className='p-2 px-4'>
                                                    <FormItem asterisk label="Picture" invalid={Boolean(errors.picture && touched.picture)} errorMessage={errors.picture as string}>
                                                        <Field name="picture" >
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Upload draggable beforeUpload={beforeUpload} fileList={values.picture}
                                                                    onChange={(files) => form.setFieldValue(field.name, files)}
                                                                    onFileRemove={(files) => form.setFieldValue(field.name, files)} uploadLimit={1} >
                                                                    <div className="my-7 text-center">
                                                                        <div className="text-6xl mb-4 flex justify-center">
                                                                            <FcImageFile />
                                                                        </div>
                                                                        <p className="font-semibold">
                                                                            <span className="text-gray-800 dark:text-white">
                                                                                Drop your image here, or{' '}
                                                                            </span>
                                                                            <span className="text-blue-500">browse</span>
                                                                        </p>
                                                                        <p className="mt-1 opacity-60 dark:text-white">
                                                                            Support: jpeg, png, gif
                                                                        </p>
                                                                    </div>
                                                                </Upload>
                                                            )}
                                                        </Field>
                                                    </FormItem>
                                                    {/* <FormItem
                                                        asterisk
                                                        label="Upload"
                                                        invalid={Boolean(
                                                            errors.picture && touched.picture
                                                        )}
                                                        errorMessage={errors.picture as string}
                                                    >
                                                        <Field name="upload">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Upload
                                                                    draggable
                                                                    beforeUpload={beforeUpload}
                                                                    fileList={values.picture}
                                                                    onChange={(files) =>
                                                                        form.setFieldValue(field.name, files)
                                                                    }
                                                                    onFileRemove={(files) =>
                                                                        form.setFieldValue(field.name, files)
                                                                    }
                                                                />
                                                            )}
                                                        </Field>
                                                    </FormItem> */}
                                                </div>
                                            </div>

                                        </div>
                                    </div>
                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-7 mt-7 "  >
                                        <div>
                                            <div className={`border rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit bg-[rgb(227,231,236)] rounded-t-md p-2 ${theme.mode == "dark" ? "text-black" : null}`}>User Address</div>
                                                <FormItem className='p-2 px-4' label="" >
                                                    <Field name="address.street1">
                                                        {({ field }: FieldProps<any>) => (
                                                            <FormItem className='mb-4' asterisk label="Street 1" invalid={errors.address?.street1 && touched.address?.street1} errorMessage={errors.address?.street1}>
                                                                <Input {...field} placeholder="Street 1" size='md' />
                                                            </FormItem>
                                                        )}
                                                    </Field>

                                                    <Field name="address.street2">
                                                        {({ field }: FieldProps<any>) => (
                                                            <FormItem className='mb-4' label="Street 2" invalid={errors.address?.street2 && touched.address?.street2} errorMessage={errors.address?.street2}>
                                                                <Input {...field} placeholder="Street 2" size='md' />
                                                            </FormItem>
                                                        )}
                                                    </Field>

                                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-4">

                                                        <Field name="address.country">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="Country" invalid={errors.address?.country && touched.address?.country} errorMessage={errors.address?.country}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={countryName}
                                                                        value={countryName.find((option: any) => option.value === values.address.country)}
                                                                        onChange={(option) => {
                                                                            if (option) {
                                                                                form.setFieldValue(field.name, option?.value)
                                                                                form.setFieldValue('address.state', '');
                                                                                form.setFieldValue('address.city', '');
                                                                                getStateList(option?.value)
                                                                            } else {
                                                                                customToast.warning("Please select a country")
                                                                            }
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.state">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="State" invalid={errors.address?.state && touched.address?.state} errorMessage={errors.address?.state}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={stateName}
                                                                        value={stateName.find((option: any) => option.value === values.address.state)}
                                                                        onChange={(option) => {
                                                                            if (touched.address?.country && option) {
                                                                                customToast.warning("Please select a country")
                                                                            } else if (option) {
                                                                                form.setFieldValue(field.name, option?.value)
                                                                                form.setFieldValue('address.city', '');
                                                                                setCityName([])
                                                                                getCityList(option?.value)
                                                                            }
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.city">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="City" invalid={errors.address?.city && touched.address?.city} errorMessage={errors.address?.city}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={cityName}
                                                                        value={cityName.find((option: any) => option.value === values.address.city)}
                                                                        onChange={(option) => {
                                                                            form.setFieldValue(field.name, option?.value)
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.zipCode">
                                                            {({ field }: FieldProps<any>) => (
                                                                <FormItem asterisk label="Zip Code" invalid={errors.address?.zipCode && touched.address?.zipCode} errorMessage={errors.address?.zipCode}>
                                                                    <Input {...field} placeholder="Zip Code" size='md' />
                                                                </FormItem>
                                                            )}
                                                        </Field>
                                                    </div>
                                                </FormItem>
                                            </div>
                                        </div>
                                        <div>
                                            <div className={`border  rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit bg-[rgb(227,231,236)] rounded-t-md p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Contact Deatils</div>
                                                <div className='p-2 px-4'>
                                                    <FormItem asterisk label="Email Address" invalid={errors.email && touched.email} errorMessage={errors.email}>
                                                        <Field type="email" name="email" placeholder="Email Address" component={Input} size='md' />
                                                    </FormItem>
                                                    <FormItem
                                                        asterisk
                                                        label="Phone Number"
                                                        invalid={
                                                            (errors.dialCode &&
                                                                touched.dialCode) ||
                                                            (errors.phoneNumber &&
                                                                touched.phoneNumber)
                                                        }
                                                        errorMessage="Please enter your phone number"
                                                    >
                                                        <InputGroup>
                                                            <Field name="dialCode">

                                                                {({
                                                                    field,
                                                                    form,
                                                                }: FieldProps) => (
                                                                    <Select<CountryOption>
                                                                        className="min-w-[150px]"
                                                                        placeholder="Dial Code"
                                                                        components={{
                                                                            Option: PhoneSelectOption,
                                                                            SingleValue:
                                                                                PhoneControl,
                                                                        }}
                                                                        field={field}
                                                                        form={form}
                                                                        options={countryList}
                                                                        value={countryList.filter(
                                                                            (country) =>
                                                                                country.dialCode ===
                                                                                values.dialCode
                                                                        )}
                                                                        onChange={(country) =>
                                                                            form.setFieldValue(
                                                                                field.name,
                                                                                country?.dialCode
                                                                            )
                                                                        }
                                                                    />
                                                                )}

                                                            </Field>
                                                            <Field name="phoneNumber">
                                                                {({
                                                                    field,
                                                                    form,
                                                                }: FieldProps) => {
                                                                    return (
                                                                        <NumericFormatInput
                                                                            form={form}
                                                                            field={field}
                                                                            customInput={
                                                                                NumberInput as ComponentType
                                                                            }
                                                                            placeholder="Phone Number"
                                                                            onValueChange={(
                                                                                e
                                                                            ) => {
                                                                                form.setFieldValue(
                                                                                    field.name,
                                                                                    e.value
                                                                                )
                                                                            }}
                                                                        />
                                                                    )
                                                                }}
                                                            </Field>
                                                        </InputGroup>
                                                    </FormItem>
                                                </div>
                                            </div>
                                            <div className={`border mt-7 rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl rounded-t-md mb-5 h-fit bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Status</div>
                                                <FormItem className='p-2 px-4' asterisk label="Status" invalid={errors.status && touched.status} errorMessage={errors.status}>
                                                    <Field name="status">
                                                        {({ field, form }: FieldProps<any>) => (
                                                            <Radio.Group value={values.status} onChange={(val) => form.setFieldValue(field.name, val)}>
                                                                {statusOptions.map(status => (
                                                                    <Radio key={status.value} value={status.value}>{status.label}</Radio>
                                                                ))}
                                                            </Radio.Group>
                                                        )}
                                                    </Field>
                                                </FormItem>
                                            </div>
                                        </div>
                                    </div>


                                    <FormItem className='text-center mt-7'>
                                        <Button
                                            type="reset"
                                            className="ltr:mr-7 rtl:ml-7"
                                            onClick={() => {
                                                resetForm()
                                                navigate('/auth/users/allUsers')
                                            }}
                                        >
                                            Reset
                                        </Button>
                                        <Button variant="solid" type="submit">
                                            Submit
                                        </Button>
                                    </FormItem>

                                </FormContainer>
                            </Form>
                        )}
                    </Formik>
                </div>
            </div>
        </>
    )
}

export default UserForm
